﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.TBI;
using AntiXssEnc = Microsoft.Security.Application;

public partial class TBI_Instruments_StationDivisionAuthorization : System.Web.UI.Page
{
    #region Actions, Error Codes, Descriptions Shared with the ASPX

    public static readonly string SentillionReasonDesc = "The sentillion components are not functioning. They are either missing or should be reinstalled.";
    public static readonly string SentillionReasonCode = "sentillion";
    public static readonly string ContextReasonDesc = "It is possible that user context sharing has been disabled by CPRS";
    public static readonly string ContextReasonCode = "context";
    public static readonly string LoginReasonDesc = "VistALink Login failed. There is either a problem connecting to the service or you are not authorized.";
    public static readonly string LoginReasonCode = "login";
    public static readonly string RequestStationAction = "requestStation";
    public static readonly string ConfirmAction = "confirm";
    public static readonly string ActionQueryField = "action";
    public static readonly string ReasonQueryField = "reason";
    public static readonly string VistaQueryField = "vista";
    public static readonly string InvalidDivisionErrorSnippet = "Division specified is invalid for user.";
    public static readonly string InvalidDivisionPleasantErrorPrefix = "The VIA service reports that you are not authorized for division ";
    public static readonly string DivisionNotSupportedFormat = "Division [{0}] not supported";
    public static readonly string SuccessMessage = "SUCCESS";
    public static readonly string VIASiteLookupSuccessPrefix = "The VIA service reports that your primary station is ";

    #endregion Error Codes and Descriptions Shared with the ASPX

    private TBIInstrumentsManager manager = null;
    private TBIServiceTO tbiServiceTO = null;
    private string siteCode = null;

    protected void Page_Load(object sender, EventArgs e)
    {
        if (!Page.IsPostBack)
        {
            if (Request.HttpMethod == "POST")
            {
                HandleLoggingRequest();
                return;
            }
            UpdateUserLabel();
            bool showSelection = false;
            string action = Request.QueryString[ActionQueryField];
            if (string.IsNullOrEmpty(action))
            {
                //The page originally loads with no query string variables.
                //The showSelection variable == false will cause the page to include the
                //CCOW single sign on code and diagnostics display.
                Logging.WriteLogToFile("StationDivisionAuthorization: Preparing to use CCOW to authenticate user to VistALink through VIA.");
            }
            else if (RequestStationAction.Equals(action, StringComparison.OrdinalIgnoreCase))
            {
                //If the division could not be determined then the RequestStationAction
                //flag is set and the Action we need to take is to Request that the user
                //select their appropriate Station.
                SetError(Request.QueryString[ReasonQueryField]);
                showSelection = true;
            }
            else if (ConfirmAction.Equals(action, StringComparison.OrdinalIgnoreCase))
            {
                //The Sentillion components were found and user context is shared. We
                //need to Confirm that the VIA service is available and that the division
                //we discovered will work for this user.
                string vistaID = Request.QueryString[VistaQueryField];
                if (!string.IsNullOrEmpty(vistaID))
                {
                    Logging.WriteLogToFile("StationDivisionAuthorization: CCOW found the following Vista ID: " + vistaID);
                    int found = vistaID.LastIndexOf('^');
                    if (-1 != found && vistaID.Length > found)
                    {
                        siteCode = vistaID.Substring(found + 1);
                        //siteCode = "442DT";//uncomment this line to test login failure
                        AttemptVIALogin();
                        showSelection = true;
                    }
                }
                else
                {
                    Logging.WriteLogToFile("StationDivisionAuthorization: CCOW was accessed but was unable to determine a Vista ID.");
                }
            }
            DefaultCCOWContent.Visible = !showSelection;//CCOW diagnostics
            DefaultJavascript.Visible = !showSelection;//CCOW javascript
            SelectionContent.Visible = showSelection;//Division Selection Dialog (if Single Sign On fails/aborts)
        }
    }

    /// <summary>
    /// Logs a javascript-generated log message to the database
    /// </summary>
    private void HandleLoggingRequest()
    {
        StreamReader reader = new StreamReader(Request.InputStream);
        string logMessage = reader.ReadToEnd();
        Logging.WriteLogToFile(string.Format("StationDivisionAuthorization: Message from {0} on {1} [{2}]: {3}",
            Request.LogonUserIdentity.Name, Request.UserHostName, Request.UserHostAddress, logMessage));
    }

    /// <summary>
    /// Requires that the page's siteCode variable is set prior to calling.
    /// </summary>
    private void AttemptVIALogin()
    {
        Logging.WriteLogToFile("StationDivisionAuthorization: Attempting VIA Login with site code = " + siteCode);
        TBIServiceTo.siteCode = siteCode;
        string error;
        var success = Manager.TestVIALogin(TBIServiceTo, out error);
        if (success)
        {
            Session["StationDivision"] = siteCode;
            Response.Redirect("PatientInstrumentsMain.aspx", true);
        }
        else
        {
            if (error.IndexOf(InvalidDivisionErrorSnippet, StringComparison.OrdinalIgnoreCase) != -1)
            {
                error = InvalidDivisionPleasantErrorPrefix + siteCode;
            }
            string temp = string.Format(DivisionNotSupportedFormat, siteCode);
            if (error.IndexOf(temp, StringComparison.OrdinalIgnoreCase) != -1)
            {
                error = temp;
            }
            SetError(LoginReasonCode, error);
        }
    }

    /// <summary>
    /// Just show something so that the user label is not empty
    /// </summary>
    private void UpdateUserLabel()
    {
        Label labelUser = (Label)Master.FindControl("LabelCurrentUser");
        labelUser.Text = "Validating...";
    }

    /// <summary>
    /// Singleton getter of a TBIServiceTO object for this page instance
    /// </summary>
    private TBIServiceTO TBIServiceTo
    {
        get
        {
            if (null == tbiServiceTO)
                tbiServiceTO = BuildTempTBIServiceTO(this);
            return tbiServiceTO;
        }
    }

    /// <summary>
    /// Singleton getter of a TBIInstrumentsManager object for this page instance
    /// </summary>
    private TBIInstrumentsManager Manager
    {
        get
        {
            if (null == manager)
                manager = new TBIInstrumentsManager();
            return manager;
        }
    }

    /// <summary>
    /// Enables calling SetError without a secondary, extended error
    /// </summary>
    /// <param name="error">The primary error to display</param>
    private void SetError(string error)
    {
        SetError(error, null);
    }

    /// <summary>
    /// Sets descriptions on error labels to present to the user.
    /// Displaying an error also means we need input from the user. So this method
    /// also sets up the station selection list box and displays instructions.
    /// </summary>
    /// <param name="error">The primary error to display</param>
    /// <param name="extendedError">The secondary error to display</param>
    private void SetError(string error, string extendedError)
    {
        ExtendedErrorLabel.Text = AntiXssEnc.Encoder.HtmlEncode(extendedError);
        if (SentillionReasonCode.Equals(error, StringComparison.OrdinalIgnoreCase))
        {
            ErrorLabel.Text = AntiXssEnc.Encoder.HtmlEncode(SentillionReasonDesc);
        }
        else if (ContextReasonCode.Equals(error, StringComparison.OrdinalIgnoreCase))
        {
            ErrorLabel.Text = AntiXssEnc.Encoder.HtmlEncode(ContextReasonDesc);
        }
        else if (LoginReasonCode.Equals(error, StringComparison.OrdinalIgnoreCase))
        {
            ErrorLabel.Text = AntiXssEnc.Encoder.HtmlEncode(LoginReasonDesc);
        }
        else
        {
            ErrorLabel.Text = "Unknown Error: " + AntiXssEnc.Encoder.HtmlEncode(error);
        }

        if (string.IsNullOrEmpty(siteCode))
        {
            string temp = Manager.LookupSiteCodeByServer(TBIServiceTo.ServerIP, out error);
            if (!string.IsNullOrEmpty(temp))
            {
                siteCode = temp;
            }
            if (!string.IsNullOrEmpty(error))
            {
                if (error == SuccessMessage)
                {
                    ExtendedErrorLabel.Text = AntiXssEnc.Encoder.HtmlEncode(VIASiteLookupSuccessPrefix + siteCode);
                }
                else
                {
                    ExtendedErrorLabel.Text = AntiXssEnc.Encoder.HtmlEncode(error);
                }
            }
        }

        //At this point we have access to the errors that are going to be displayed to the user, so log them
        Logging.WriteLogToFile(string.Format("StationDivisionAuthorization: Error:\r\nPrimary={0}\r\nSecondary=", ErrorLabel.Text, ExtendedErrorLabel.Text));
        StationBox.DataSource = GetMatchingStations(siteCode);
        StationBox.DataBind();
        if (!string.IsNullOrEmpty(siteCode))
        {
            //if the login fails do not attempt again automatically
            if (StationBox.Items.Count == 1 && error != LoginReasonCode)
            {
                string availableStation = (StationBox.Items[0]).Value;
                if (availableStation == siteCode)
                {
                    //there is only one available station and it matches the station sent by CPRS
                    //so instead of showing an error and a selection list, just go ahead and
                    //attempt to Login the user using that station
                    AttemptVIALogin();
                }
            }
            StationBox.SelectedValue = siteCode;
            InstructionsLabel.Text = "To try signing on to another division, select it from the list then click Submit.";
        }
        else
        {
            InstructionsLabel.Text = "To try signing on to a division, select it from the list then click Submit.";
        }
    }

    /// <summary>
    /// ViewModel class for displaying available stations
    /// </summary>
    public class Station
    {
        public string Code { get; private set; }
        public string Description { get; private set; }
        public Station(string code, string description)
        {
            Code = code;
            Description = description;
        }
    }

    /// <summary>
    /// Selects the divisions matching a given station
    /// </summary>
    /// <param name="baseStation">Can either be null, a three digit station, or a five dight station.</param>
    /// <returns>Returns the base station and all divisions associated with the base station. Returns all stations if no base station is provided.</returns>
    private IEnumerable<Station> GetMatchingStations(string baseStation)
    {
        List<Station> result = new List<Station>();
        if (null == baseStation)
        {
            baseStation = "";
        }
        else if (baseStation.Length > 3)
        {
            baseStation = baseStation.Substring(0, 3);
        }

        //string sql = string.Format("select name, stationnumber from std_institution where stationnumber like '{0}%' order by stationnumber", baseStation);
        //using (SqlConnection connection = new SqlConnection(VeteransAffairs.Registries.Business.GlobalSettings.DALConnectionString))
        //{
        //    connection.Open();
        //    using (SqlCommand command = new SqlCommand(sql, connection))
        //    {
        //        using (SqlDataReader reader = command.ExecuteReader())
        //        {
        //            while (reader.Read())
        //            {
        //                string code = reader.GetString(1);
        //                result.Add(new Station(code, string.Format("{0} ({1})", reader.GetString(0), code)));
        //            }
        //        }
        //    }
        //}

        InstitutionManager institutionManager = new InstitutionManager();
        IEnumerable<STD_INSTITUTION> iList = institutionManager.Select(baseStation);
        if (iList != null)
        {
            foreach (STD_INSTITUTION i in iList)
            {
                result.Add(new Station(i.STATIONNUMBER, i.NAME));
            }
        }

        return result;
    }

    /// <summary>
    /// Helper function used to return a temporary TBIServerTO object that will be used
    /// to test whether or not the configuration provided can be used to successfully
    /// login to a VistALink server through VIA.
    /// </summary>
    /// <param name="page">Page context to pass to helper function that needs its Session information.</param>
    /// <returns>Temporary TBIServiceTO object only used to test login</returns>
    private static TBIServiceTO BuildTempTBIServiceTO(Page page)
    {
        string _inSiteCode, _inDUZ, _inDFN, _inServerIP, _inServerPort;
        int _inTBIEvaluationNoteId, _inTBIFollowUpNoteId;
        Helpers.ExtractCPRSVariables(page, out _inSiteCode, out _inDUZ, out _inDFN, out _inServerIP, out _inServerPort, out _inTBIEvaluationNoteId, out _inTBIFollowUpNoteId);
        TBIServiceTO result = Helpers.BuildTBIServiceTO(_inSiteCode, _inDUZ, _inDFN, _inServerIP, _inServerPort, _inTBIEvaluationNoteId, _inTBIFollowUpNoteId);
        return result;
    }

    protected void SubmitButton_Click(object sender, EventArgs e)
    {
        if (StationBox.SelectedIndex == -1)
            return;
        siteCode = StationBox.SelectedValue;
        AttemptVIALogin();
    }
}